///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef _SQLEXPORT_CPP
#define _SQLEXPORT_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WindowsX.H>
#include <ShellAPI.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <SQL.H>
#include <SQLExt.H>

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "../Resources/Resource.H"
#include "../../SharedClasses/CMemPool/CMemPool.H"

#include "../CSockSrvr/CSockSrvr.H"

#include "../../SharedSource/Debug.H"
#include "../../SharedSource/NSWFL.H"
#include "../../SharedSource/Common.H"
#include "../../SharedSource/CRC32.H"

#include "Init.H"
#include "Routines.H"
#include "Command.H"
#include "SQLExport.H"
#include "Console.H"

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteNull(FILE *fTarget)
{
	const char *sNullStr = "NULL";
	const int iNullStrSz = 4;

	fwrite(&iNullStrSz, sizeof(iNullStrSz), 1, fTarget);
    fwrite(sNullStr, sizeof(char), iNullStrSz, fTarget);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceSingleQuotes(const char *sBuf, char *sOutBuf, int iBufSz)
{
	int iRPos = 0;
	int iWPos = 0;

	while(iRPos < iBufSz)
	{
		if(sBuf[iRPos] == '\'')
		{
			sOutBuf[iWPos++] = '\'';
		}

		sOutBuf[iWPos++] = sBuf[iRPos++];
	}

	sOutBuf[iWPos] = '\0';

	return iWPos;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsBinary(const void *lpBuf, int iSz)
{
	int iRPos = 0;

	while(iRPos < iSz)
	{
		if(((unsigned char *)lpBuf)[iRPos] < 32 || ((unsigned char *)lpBuf)[iRPos] > 126)
		{
			return true;
		}
		iRPos++;
	}
	return false;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int Bin2Hex(const void *lpSrc, int iSz, void *lpTarget, int iMaxSz)
{
	char sHex[16];
	int iHexSz = 0;
	int iFullHexSz = 0;
	strcpy_s((char *)lpTarget, iMaxSz, "0x");
	iFullHexSz = 2; // "0x"

	int iHexPos = 0;
	while(iHexPos < iSz)
	{
		sprintf_s(sHex, sizeof(sHex), "%X", ((unsigned char *)lpSrc)[iHexPos]);
		if((iHexSz = strlen(sHex)) > 2)
		{
			//FIXME: This could cause a big blow up. Need to error and exit.
		}

		if(iHexSz < 2)
		{
			strcat_s((char *)lpTarget, iMaxSz, "0");
			iHexSz++;
		}

		strcat_s((char *)lpTarget, iMaxSz, sHex);

		iFullHexSz += iHexSz;
		iHexPos++;
	}

	((char *)lpTarget)[iFullHexSz] = '\0';

	return iFullHexSz;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ExportSQLResults(CSockSrvr *pSockSrvr, int iClient, int iID, char *Statement,
					 char *sDB, char *sDBO, char *sImportTable, char *OutFileName)
{
	FILE *TargetHandle = NULL;

	CRecordSet rsTemp;

    SQLRETURN SQLResult = 0;

    char sStatusText[MAX_STATUS_TEXT]; //Just some RAM for status text and error messages.
    char sAltTemp[MAX_STATUS_TEXT];    //Just some RAM for status text and error messages.
    char sOutColumnName[256];
	char sSQL[255];

	int iTemp = 0;
	int iInColumnNameLen = 256;
	int *iOutColumnSize = 0;
    int iOutColumnNameLen = 0;
    int *iOutDataType = 0;
    int iOutDecimalDigits = 0;
    int iOutNullable = 0;
    int iColumn = 0;

	rsTemp.bReplaceSingleQuotes = false;
	rsTemp.bThrowErrors = true;
	rsTemp.bTrimCharData = true;

	sprintf_s(sSQL, sizeof(sSQL), "USE [%s]", sDB);
	if(!CCI.cCustSQL.ExecuteNonQuery(sSQL))
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Failed to change client database context. Performance will be affected.", EVENT_WARN);
	}

	if(gbDebugMode)
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Executing SQL statement.", EVENT_NONE);
	}

	if(!CCI.cCustSQL.Execute(Statement, &rsTemp))
    {
		WriteLogEx(pSockSrvr->icClientID[iClient], "ExportSQLResults SELECT Execute failed. (See log file for more information)", EVENT_ERROR);
		WriteLogEx(pSockSrvr->icClientID[iClient], Statement, EVENT_INFO);

		//Need to do some advanced error reporting.
		if(CCI.cCustSQL.GetErrorMessage(sAltTemp, sizeof(sAltTemp), rsTemp.hSTMT))
		{
			sprintf_s(sStatusText, sizeof(sStatusText), "Error Msg: %s", sAltTemp);
			WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);
		}

		return SQL_IMPORT_RESULT_ERROR;
    }

    int iOutColCount = rsTemp.ColCount;
    int iOutRowCount = rsTemp.RowCount;

    if(iOutRowCount == 0)
    {
		if(gbDebugMode)
		{
			WriteLogEx(pSockSrvr->icClientID[iClient], "Query returned zero rows.", EVENT_NONE);
		}

        rsTemp.Close();

        return SQL_IMPORT_RESULT_ZEROROWS;
    }
    else{
        sprintf_s(sStatusText, sizeof(sStatusText),
			"Exporting transaction data. [%s.%s - %d rows, %d columns]", sDB, sImportTable, iOutRowCount, iOutColCount);
		WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_NONE);
    }

    if(fopen_s(&TargetHandle, OutFileName, "wb") != 0)
    {
        sprintf_s(sStatusText, sizeof(sStatusText),
			"TargetFile: Binary write error with (%s).", OutFileName);
		WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);

        rsTemp.Close();

        return SQL_IMPORT_RESULT_ERROR;
    }

	if(gbDebugMode)
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Writing export file header.", EVENT_NONE);
	}

	iTemp = strlen(sDB);
    fwrite(&iTemp, sizeof(iTemp), 1, TargetHandle);               // Write import database name len.
    fwrite(sDB, sizeof(char), iTemp, TargetHandle);               // Write import database name.

	iTemp = strlen(sImportTable);
	fwrite(&iTemp, sizeof(iTemp), 1, TargetHandle);               // Write import table name len.
    fwrite(sImportTable, sizeof(char), iTemp, TargetHandle);      // Write import table name.

	fwrite(&iOutColCount, sizeof(iOutColCount), 1, TargetHandle); // Write number of columns.
    fwrite(&iOutRowCount, sizeof(iOutRowCount), 1, TargetHandle); // Write number of rows.

    fwrite(&iID, sizeof(iID), 1, TargetHandle); // Write the row ID of the associated Statments row.

    //---------------( Get column Information )---

	if(gbDebugMode)
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Retreving column set information.", EVENT_NONE);
	}

	iOutDataType = (int *) gMem.Allocate( sizeof(int), iOutColCount);
	Assert(!iOutDataType, "Memory Allocation Error.");
	iOutColumnSize = (int *) gMem.Allocate( sizeof(int), iOutColCount);
	Assert(!iOutColumnSize, "Memory Allocation Error.");

    while(rsTemp.GetColInfo(iColumn + 1, sOutColumnName, iInColumnNameLen, &iOutColumnNameLen,
		&iOutDataType[iColumn], &iOutColumnSize[iColumn], &iOutDecimalDigits, &iOutNullable))
	{
		fwrite(&iOutDataType[iColumn], sizeof(iOutDataType[iColumn]), 1, TargetHandle);
		fwrite(&iOutColumnSize[iColumn], sizeof(iOutColumnSize[iColumn]), 1, TargetHandle);
		fwrite(&iOutColumnNameLen, sizeof(iOutColumnNameLen), 1, TargetHandle);
		fwrite(sOutColumnName, sizeof(char), iOutColumnNameLen, TargetHandle);

		/*
		sprintf_s(sStatusText, sizeof(sStatusText), "Col:%d SQL_TYPE_%s %d", iColumn, UCase(sOutColumnName), iOutDataType[iColumn]);
		WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);
		*/

		iColumn++;
    }

	if(gbDebugMode)
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Exporting column set data.", EVENT_NONE);
	}

	int sSQLBufAlloc = MAX_SQL_DATA_SZ * 2;
	int sSQLBuf2Alloc = MAX_SQL_DATA_SZ * 2;

	char *sSQLBuf = (char *) gMem.Allocate(sizeof(char), (MAX_SQL_DATA_SZ * 2) + 1);
	Assert(!sSQLBuf, "Memory Allocation Error.");
	char *sSQLBuf2 = (char *) gMem.Allocate(sizeof(char), (MAX_SQL_DATA_SZ * 2) + 1);
	Assert(!sSQLBuf2, "Memory Allocation Error.");

	//---------------( Get column Data )---
    while(rsTemp.Fetch())
    {
        iColumn = 0;

		EnterCriticalSection(&csTransPerSecond);
		giTransProcessed++;
		LeaveCriticalSection(&csTransPerSecond);
			
		while(iColumn < iOutColCount)
        {
            int iDataType = iOutDataType[iColumn];
            int iDataLen = iOutColumnSize[iColumn];

			//------------------------------------------------------------------------------------------
			if(		iDataType == TSQL_NTEXT       || iDataType == TSQL_TEXT             ||
					iDataType == TSQL_DATETIME    || iDataType == TSQL_SMALLDATETIME    ||
					iDataType == TSQL_CHAR        || iDataType == TSQL_SQL_VARIANT      ||
					iDataType == TSQL_VARCHAR     || iDataType == TSQL_UNIQUEIDENTIFIER)
			{
				bool bConvertToBinary = false;

				//If the data type is Text, NText or Image then only retreive a few KB's of it.
				if(iDataType == TSQL_TEXT || iDataType == TSQL_NTEXT || iDataType == TSQL_IMAGE)
				{
					iDataLen = MAX_SQL_DATA_SZ;
				}

				if(rsTemp.sColumnEx(iColumn + 1, sSQLBuf, iDataLen + 100, &iDataLen))
                {
					if(iDataType == TSQL_CHAR || iDataType == TSQL_VARCHAR || iDataType == TSQL_SQL_VARIANT)
					{
						bConvertToBinary = IsBinary(sSQLBuf, iDataLen);
					}

					//If the data type is Text, NText or Image then only retreive a few KB's of it.
					if(iDataType == TSQL_TEXT || iDataType == TSQL_NTEXT || iDataType == TSQL_IMAGE)
					{
						if(iDataLen > MAX_SQL_DATA_SZ)
						{
							iDataLen = MAX_SQL_DATA_SZ;
						}
					}

                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
						if(bConvertToBinary)
						{
							iDataLen = Bin2Hex(sSQLBuf, iDataLen, sSQLBuf2, sSQLBuf2Alloc);

							fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle); //Write the data length.
							fwrite(sSQLBuf2, sizeof(char), iDataLen, TargetHandle); //Write the data.
						}
						else{
							sSQLBuf[iDataLen] = '\0';

							iDataLen = ReplaceSingleQuotes(sSQLBuf, sSQLBuf2, iDataLen);

							iDataLen = (iDataLen + 2); // To make room for the single quotes
							fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle); //Write the data length.
							iDataLen = (iDataLen - 2); // Because we only write the origional trimmed length

							fwrite("'", sizeof(char), 1, TargetHandle); // Wrap the data in single quotes
							fwrite(sSQLBuf2, sizeof(char), iDataLen, TargetHandle); //Write the data.
							fwrite("'", sizeof(char), 1, TargetHandle); // Wrap the data in single quotes
						}
					}
				}
				else{
	                sprintf_s(sStatusText, sizeof(sStatusText),
						"Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);
				}
            }
			//------------------------------------------------------------------------------------------
			else if(iDataType == TSQL_BINARY || iDataType == TSQL_VARBINARY /*iDataType == TSQL_TIMESTAMP*/
				|| iDataType == TSQL_NVARCHAR || iDataType == TSQL_NCHAR || iDataType == TSQL_IMAGE)
            {
                if(rsTemp.BinColumnEx(iColumn + 1, sSQLBuf, MAX_SQL_DATA_SZ, &iDataLen))
				{
                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
						iDataLen = Bin2Hex(sSQLBuf, iDataLen, sSQLBuf2, sSQLBuf2Alloc);

						fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle); //Write the data length.
						fwrite(sSQLBuf2, sizeof(char), iDataLen, TargetHandle); //Write the data.
					}
                }
				else{
	                sprintf_s(sStatusText, sizeof(sStatusText),
						"Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);
				}
            }
			//------------------------------------------------------------------------------------------
			else if(iDataType == TSQL_BIGINT     || iDataType == TSQL_BIT        ||
					iDataType == TSQL_DECIMAL    || iDataType == TSQL_FLOAT      ||
					iDataType == TSQL_INT        || iDataType == TSQL_MONEY      ||
					iDataType == TSQL_NUMERIC    || iDataType == TSQL_REAL       ||
					iDataType == TSQL_SMALLINT   || iDataType == TSQL_SMALLMONEY ||
					iDataType == TSQL_TINYINT)
            {
                if(rsTemp.sColumnEx(iColumn + 1, sSQLBuf, MAX_SQL_DATA_SZ, &iDataLen))
				{
                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
						fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle);  //Write the data length.
						fwrite(sSQLBuf, sizeof(char), iDataLen, TargetHandle); //Write the data.
					}
                }
				else{
	                sprintf_s(sStatusText, sizeof(sStatusText),
						"Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);
				}
            }
			//------------------------------------------------------------------------------------------
			else{
                sprintf_s(sStatusText, sizeof(sStatusText),
					"Unknown SQL data type. Table: %s, Select column: %d, Type: %d", sImportTable, iColumn, iDataType);
				WriteLogEx(pSockSrvr->icClientID[iClient], sStatusText, EVENT_ERROR);

				gMem.Free(sSQLBuf);
				gMem.Free(sSQLBuf2);
				gMem.Free(iOutColumnSize);
				gMem.Free(iOutDataType);

				rsTemp.Close();

				if(TargetHandle) fclose(TargetHandle);

			    return SQL_IMPORT_RESULT_ERROR;
			}
			//------------------------------------------------------------------------------------------

            iColumn++;
        }
    }

	if(gbDebugMode)
	{
		WriteLogEx(pSockSrvr->icClientID[iClient], "Ending the export successfully.", EVENT_NONE);
	}

	gMem.Free(sSQLBuf);
	gMem.Free(sSQLBuf2);
	gMem.Free(iOutColumnSize);
	gMem.Free(iOutDataType);

	rsTemp.Close();

    if(TargetHandle) fclose(TargetHandle);

	return SQL_IMPORT_RESULT_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
